/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#pragma once

#include <algorithm>
#include <stddef.h>
#include <stdexcept>
#include <stdint.h>
#include <string.h>

#if _WIN32
	#define WIN32_LEAN_AND_MEAN 1
	#include <Windows.h>
#else
	#include <fcntl.h>
	#include <sys/mman.h>
#endif

#include "Defer.h"
#include "ParallelFor.h"
#include "Point.h"



namespace DbgTablePainter
{
	struct Pixel
	{
		uint8_t B;
		uint8_t G;
		uint8_t R;
		uint8_t A;
	};

	// void fill(char* ptr)
	template<typename LambdaT> static void WriteNewFile(const char* filename, size_t length, LambdaT&& fill)
	{
#if _WIN32
		HANDLE hFile = CreateFileA(filename, GENERIC_READ | GENERIC_WRITE, 0, nullptr, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0);
		if (hFile == INVALID_HANDLE_VALUE)
			throw std::runtime_error("Unable to create file.");
		DEFER { CloseHandle(hFile); };
		HANDLE hMap = CreateFileMappingW(hFile, nullptr, PAGE_READWRITE | SEC_COMMIT, (DWORD)((DWORDLONG)length >> 32), (DWORD)length, nullptr);
		if (hMap == INVALID_HANDLE_VALUE)
			throw std::runtime_error("Unable to create file mapping.");
		DEFER { CloseHandle(hMap); };
		void* ptr = MapViewOfFile(hMap, FILE_MAP_WRITE, 0, 0, 0);
		if (!ptr)
			throw std::runtime_error("Unable to create file view.");
		DEFER { UnmapViewOfFile(ptr); };
#else
		int fileDescriptor = open(filename, O_CREAT | O_RDWR, 0644);
		if (fileDescriptor == -1)
			throw std::runtime_error("Unable to create file.");
		DEFER { close(fileDescriptor); };
		if (ftruncate(fileDescriptor, length))
			throw std::runtime_error("Unable to resize file.");
		void* ptr = mmap(nullptr, length, PROT_WRITE, MAP_SHARED, fileDescriptor, 0);
		DEFER { munmap(ptr, length); };
		if (ptr == MAP_FAILED)
			throw std::runtime_error("Unable to \"mmap\".");
#endif
		fill((char*)ptr);
	}

	static inline void WriteInt16(char* ptr, uint16_t data)
	{
		*ptr = (char)(data & 0xff);
		ptr[1] = (char)(data >> 8);
	}

	static inline void WriteInt32(char* ptr, uint32_t data)
	{
		*ptr = (char)(data & 0xff);
		ptr[1] = (char)(data >> 8);
		ptr[2] = (char)(data >> 16);
		ptr[3] = (char)(data >> 24);
	}

	// void fill(Pixel* ptr)
	template<typename LambdaT> static void Write32BitBMP(const char* filename, size_t imageWidth, size_t imageHeight, LambdaT&& fill)
	{
		const size_t headerSize = 54;
		size_t pixelSize = imageWidth * imageHeight * 4;
		WriteNewFile(filename, headerSize + pixelSize, [&](char* data) {
			WriteInt16(data + 0, 0x4D42); // Identifier of file type.
			WriteInt32(data + 2, (uint32_t)(headerSize + pixelSize));
			WriteInt32(data + 6, 0); // Unused
			WriteInt32(data + 10, (uint32_t)headerSize);
			WriteInt32(data + 14, 40); // Extended header size.
			WriteInt32(data + 18, (uint32_t)imageWidth);
			WriteInt32(data + 22, (uint32_t)imageHeight);
			WriteInt16(data + 26, 1);  // 1 Plane (???)
			WriteInt16(data + 28, 32); // 32 Bit/Pixel
			WriteInt32(data + 34, (uint32_t)pixelSize);
			WriteInt32(data + 38, 2835); // Pixels per meter (like DPI) in x
			WriteInt32(data + 42, 2835); // Pixels per meter (like DPI) in y
			memset(data + 46, 0, 8);     // Unused
			fill((Pixel*)(data + headerSize));
		});
	}

	// Point3D getColor(Point4D pos)
	template<typename LambdaT> static void Visualize4DRange(Point4DInt counts, const char* filename, LambdaT&& getColor)
	{
		const size_t margin = 1;
		size_t imageWidth = counts.Z * (counts.X + margin) - margin;
		size_t imageHeight = counts.M * (counts.Y + margin) - margin;
		Write32BitBMP(filename, imageWidth, imageHeight, [&](Pixel* pixels) {
			// for (int y = 0; y < counts.Y; ++y)
			ParallelFor((size_t)counts.Y, [&](size_t y) {
				for (size_t m = 0; m < (size_t)counts.M; ++m)
					for (size_t z = 0; z < (size_t)counts.Z; ++z)
					{
						size_t imageX = z * (counts.X + margin);
						size_t imageY = m * (counts.Y + margin) + y;
						Pixel* rowPtr = pixels + imageY * imageWidth + imageX;
						for (size_t x = 0; x < (size_t)counts.X; ++x)
						{
							Point3D color = getColor(Point4DInt((int32_t)x, (int32_t)y, (int32_t)z, (int32_t)m));
							Pixel* pixelPtr = rowPtr + x;
							pixelPtr->R = (uint8_t)std::max(0.0, std::min(255.0, color.X * 255.0 + 0.5));
							pixelPtr->G = (uint8_t)std::max(0.0, std::min(255.0, color.Y * 255.0 + 0.5));
							pixelPtr->B = (uint8_t)std::max(0.0, std::min(255.0, color.Z * 255.0 + 0.5));
							pixelPtr->A = 255;
						}
					}
			});
		});
	}
} // namespace DbgTablePainter